using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Data;
using System.Data.Common;
using System.Data.Odbc;

namespace DBClassWrapper
{

    public enum TReaderType
    {
        rtNone = 0,
        rtDotNet = 1,
        rtAdo = 2,
        rtSQL = 3,
        rtOle = 4, 
        rtODBC = 5,
        rtOracle = 6

    }

    static public class CommonFuncs
    {

        static public string ReaderTypeToString(TReaderType rt)
        {
            string s = "";

            switch(rt)
            {
                case TReaderType.rtNone :
                    s = "";
                    break;
                case TReaderType.rtDotNet :
                    s = "DataReader";
                    break;
                case TReaderType.rtAdo :
                    s = "DBDataReader";
                    break;
                case TReaderType.rtSQL :
                    s = "SqlDataReader";
                    break;
                case TReaderType.rtOle :
                    s = "OleDataReader";
                    break;
                case TReaderType.rtODBC :
                    s = "OdbcDataReader";
                    break;
                case TReaderType.rtOracle :
                    s = "OracleDataReader";
                    break;
            }

            return s;
        }

        static public string DataTypeStringToString(string DataType)
        {
            string s = "";

            switch (DataType)
            {
                case "System.String" :
                    s = "string";
                    break;
                case "System.Boolean" :
                    s = "boolean";
                    break;
                case "System.Byte" :
                    s = "byte";
                    break;
                case "System.Int64":
                    s = "int";
                    break;
                case "System.Int32" :
                    s = "int";
                    break;
                case "System.Int16" :
                    s = "int";
                    break;
                case "System.Double" :
                    s = "double";
                    break;
                case "System.DateTime" :
                    s = "DateTime";
                    break;
                case "System.TimeSpan" :
                    s= "DateTime";
                    break;
                default :
                    s = "string";
                    break;
            }

            return s;
        }

        static public string DataTypeStringToDBString(string aDataType, string fieldname)
        {
            string s = "";

            switch (aDataType)
            {
                case "System.Boolean":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  "
                        + "false" + ":" + "Convert.ToBoolean(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.Byte":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "0" + ":" + "Convert.ToByte(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.Int64":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "0" + ":" + "Convert.ToInt32(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.Int32":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "0" + ":" + "Convert.ToInt32(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.Int16":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "0" + ":" + "Convert.ToInt32(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.Double":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "0.0D" + ":" + "Convert.ToDouble(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.DateTime":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "DateTime.Now" + ":" + "Convert.ToDateTime(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                case "System.TimeSpan":
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  " 
                        + "DateTime.Now" + ":" + "Convert.ToDateTime(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
                default:
                    s = "(Convert.IsDBNull(CurrentReader[\"" + fieldname + "\"]) == true ?  "
                        + "\"\"" + ":" + "Convert.ToString(CurrentReader[\"" + fieldname + "\"])" + ")";
                    break;
            }

            return s;
        }

        static public string ExtractParamValue(string param)
        {
            string res = "";

            if (param.IndexOf('=') > 0)
            {
                // 01234567890
                // -v="Welco"
                res = param.Substring(param.IndexOf('=') + 1, (param.Length - param.IndexOf('=') -1));
                res = res.Replace('"', ' ');
                res = res.Trim();
            }

            return res;
        }

        static public int FindParam(string[] p, string param)
        {
            int res = -1;


            for (int i = 0; i < p.Length; i++)
            {
                if (p[i].IndexOf(param) == 0)
                {
                    res = i;
                    break;
                }
            }
            return res;           
        }

        static public string Help()
        {

            return
                "DBClassWrapper [-h -o -u -p - t -R -P]\n " +
                "  Wraps tables in C# classes, which saves you time. I hope.\n" +
                "  (c) 2007 Arthur Hoogervorst\n" +
                "Options:  -h=Prints this screen\n" +
                "          -o=YourConnection\n" +
                "          -u=Username [\"\" = default]\n" +
                "          -p=password [\"\" = default]\n" +
                "          -t=tablename\n" +
                "          -R=[0..6]  \n" +
                "                    0 : No SetFields datareader\n" +
                "                    1 : DataReader\n" +
                "                    2 : DBDataReader\n" +
                "                    3 : SqlDataReader\n" +
                "                    4 : OleDBDataReader\n" +
                "                    5 : OdbcDataReader [default]\n" +
                "                    6 : OracleDataReader\n" +
                "          -P=Prettifier (TM)\n";
            
        }

        public static string Capitalize(string value)
        {
            return System.Globalization.CultureInfo.CurrentCulture.TextInfo.ToTitleCase(value);
        }

        static public string Prettify(string astring)
        {
            string res = "";

            string[] p = astring.Split('_', ' ', '-');

            for(int i = 0; i < p.Length; i++)
            {
                p[i] = Capitalize(p[i]);
            }

            res = String.Join("", p);


            return res;
        }
    }

    public class TDataCollector
    {
        private string fconnectionstring = "";
        private string fuserid = "";
        private string fpassword = "";

        private OdbcConnection fcurrentconnection = new OdbcConnection();
        private DataTable finternaltable = new DataTable();
        private DataTable finternaldatatypes = new DataTable();

        private bool fprettifiertm = false; // HAHA!
        private TReaderType freadertype = TReaderType.rtODBC;

        private void InitInternals()
        {
            finternaltable.Clear();
            finternaldatatypes.Clear();
        }
        
        private int DoConnection()
        {
            int res = 0;
            try
            {
                if (fcurrentconnection.State != ConnectionState.Closed)
                {
                    fcurrentconnection.Close();
                }

                fcurrentconnection.ConnectionString = "DSN=" + fconnectionstring + ";" +
                    (fuserid == "" ? "" : "UID=" + fuserid + ";") +
                    (fpassword == "" ? "" : "PWD=" + fpassword + ";");
                fcurrentconnection.Open();
                res = (fcurrentconnection.State == ConnectionState.Open ? 1 : 0);
            }
            catch
            {
                res = -1;
            }
            return res;
        }

        public string ConnectionString
        {
            get { return fconnectionstring; }
            set { fconnectionstring = value; }
        }

        public TReaderType ReaderType
        {
            get { return freadertype; }
            set { freadertype = value; }
        }

        public bool Prettifier
        {
            get { return fprettifiertm; }
            set { fprettifiertm = value; }
        }

        public OdbcConnection Connection
        {
            get { return fcurrentconnection; }
        }
        
        public TDataCollector(string connectionstring, string auserid, string apassword)
        {
            InitInternals();
            fconnectionstring = connectionstring;
            fuserid = auserid;
            fpassword = apassword;
        }

        private int GetDataForTable(string atablename)
        {
            int res = 0;

            string[] a = { "", "", atablename};

            try
            {
                finternaltable = fcurrentconnection.GetSchema("COLUMNS", a);
                finternaldatatypes = fcurrentconnection.GetSchema("DATATYPES");
                res = ((finternaltable.Rows.Count > 0 && finternaldatatypes.Rows.Count > 0) ? 1 : 0);               
            }
            catch
            {
                res = -1;
            }

            return res;
        }

        private string FindDataType(int aninteger)
        {
            string res = "";

            DataRow[] nrow = finternaldatatypes.Select("SQLType = " + Convert.ToString(aninteger));

            if (nrow.Length > 0)
            {
                res = nrow[0]["DataType"].ToString();
            }
            return res;

        }

        private string GetPrivateFields()
        {
            string s = "";
            for (int i = 0; i < finternaltable.Rows.Count; i++)
            {
                s += "private " + CommonFuncs.DataTypeStringToString(
                    this.FindDataType(Convert.ToInt32(finternaltable.Rows[i]["DATA_TYPE"]))) 
                    +" f" + finternaltable.Rows[i]["COLUMN_NAME"].ToString() + ";\n\r";
            }

            return s;
        }

        private string GetPublicFields()
        {
            string s = "";

            for (int i = 0; i < finternaltable.Rows.Count; i++)
            {
                s += "public " + CommonFuncs.DataTypeStringToString(
                    this.FindDataType(Convert.ToInt32(finternaltable.Rows[i]["DATA_TYPE"])))
                    + " " + (this.fprettifiertm == true? 
                       CommonFuncs.Prettify(finternaltable.Rows[i]["COLUMN_NAME"].ToString()) :
                        finternaltable.Rows[i]["COLUMN_NAME"].ToString())
                    + "{\n\r get { return f" + finternaltable.Rows[i]["COLUMN_NAME"].ToString() 
                    + ";}"
                    + "\n\r set { f" + finternaltable.Rows[i]["COLUMN_NAME"].ToString() 
                    + " = value;}}\n\r\n\r";                  
            }

            return s;
        }

        private string GetPublicSetFieldsMethod()
        {
            string s = "";

            s += "public void SetFields(" 
                + CommonFuncs.ReaderTypeToString(this.freadertype) + " CurrentReader){\n\r";
            for (int i = 0; i < finternaltable.Rows.Count; i++)
            {
                s += "this.f" + finternaltable.Rows[i]["COLUMN_NAME"].ToString() + " = " +
                    CommonFuncs.DataTypeStringToDBString(                    
                      this.FindDataType(Convert.ToInt32(finternaltable.Rows[i]["DATA_TYPE"])),
                    finternaltable.Rows[i]["COLUMN_NAME"].ToString()) + ";\n\r";
            }

            s += "}\n\r";

            return s;
        }

        public string CollectData(string table)
        {
            string ress = "";
            InitInternals();
            DoConnection();
            try
            {
                int res = GetDataForTable(table);

                if (res > 0)
                {
                    ress += "public class DB_" + table + "{\n\r";
                    ress += this.GetPrivateFields();
                    ress += "// public properties\n\r";
                    ress += this.GetPublicFields();
                    if (freadertype != TReaderType.rtNone)
                    {
                        ress += GetPublicSetFieldsMethod();
                    }
                    ress += "}\n\r";

                }
            }
            catch 
            {

            }

            return ress;
        }
    }
}
